#define DEFERRED

#include "/lib/all_the_libs.glsl"

in vec2 texcoord;
flat in vec3 LightColorDirect; // This needs to be initialized in the vertex stage of the pass

#include "/generic/water.glsl"
#include "/generic/shadow/main.glsl"
#include "/generic/gtao.glsl"
#include "/generic/sky.glsl"
#include "/generic/shadow/vl.glsl"
#include "/generic/fog.glsl"
#include "/generic/clouds.glsl"
#include "/generic/reflections.glsl"
#include "/generic/shadow/rsm.glsl"

#ifdef SPATIAL_DENOISING
    /* DRAWBUFFERS:03 */
#else
    /* DRAWBUFFERS:0 */
#endif

layout(location = 0) out vec4 Color;

#ifdef SPATIAL_DENOISING
    layout(location = 1) out vec4 GISpatial;
#endif

// Taken from spectrum: https://github.com/zombye/spectrum
vec3 refract2(vec3 I, vec3 N, vec3 NF, float eta) {
    float NoI = dot(N, I);
    float k = 1.0 - eta * eta * (1.0 - NoI * NoI);
    if (k < 0.0) {
        return vec3(0.0);
    } else {
        float sqrtk = sqrt(k);
        vec3 R = (eta * dot(NF, I) + sqrtk) * NF - (eta * NoI + sqrtk) * N;
        return normalize(R * sqrt(abs(NoI)) + eta * I);
    }
}

void main() {
    float Depth = texture(depthtex0, texcoord).x;
    Positions Pos = get_positions(texcoord, Depth, true);
    bool IsHand = Depth <= 0.56;
    if (IsHand) {
        Depth = Depth * 2 - 1;
        Depth /= MC_HAND_DEPTH;
        Depth = Depth * 0.5 + 0.5;
    }

    float Dither = dither(gl_FragCoord.xy);

    if (Depth < 1) {
        vec4 Data = texture(colortex1, texcoord);
        vec4 Data2 = texture(colortex2, texcoord);
        vec2 UnpackX = unpackUnorm2x8(Data.x);
        vec2 UnpackY = unpackUnorm2x8(Data.y);
        float Material = UnpackY.y * 255;
        vec3 Normal = decodeUnitVector(unpackUnorm2x8(Data.w) * 2 - 1);
        Normal = player_view(Normal);
        vec3 FlatNormal = decodeUnitVector(unpackUnorm2x8(Data2.w) * 2 - 1);
        FlatNormal = player_view(FlatNormal);

        float Depth1 = texture(depthtex1, texcoord).x;
        vec3 ViewPos1 = screen_view(vec3(Pos.Screen.xy, Depth1), true);
        if (Depth != Depth1 && Depth1 < 1) {
            vec3 RefractedDir = refract2(Pos.ViewN, Normal, FlatNormal, 0.7518);

            vec3 RefractedPos = ViewPos1 + RefractedDir * distance(Pos.View, ViewPos1);
            vec3 ScreenPosRef = view_screen(RefractedPos, true);

            vec4 NewDepths = textureGather(depthtex1, ScreenPosRef.xy, 0);
            float NewDepthClosest = min_component(NewDepths);
            if (NewDepthClosest < Depth)
                ScreenPosRef = Pos.Screen;
            else {
                Depth1 = lerp(NewDepths.x, NewDepths.y, NewDepths.z, NewDepths.w, ScreenPosRef.xy);
                ViewPos1 = screen_view(vec3(Pos.Screen.xy, Depth1), true);
            }

            Color = texture(colortex0, ScreenPosRef.xy);
        }
        else {
            Color = texture(colortex0, Pos.Screen.xy);
        }

        if (Material == MATERIAL_WATER && isEyeInWater == 0) {
            vec3 PlayerPos1 = view_player(ViewPos1);
            Color.rgb = do_water_vl(Pos.Player, PlayerPos1, Pos.PlayerN, Color.rgb, Depth1, 4, VL_WATER_RT);
        }

        float Smoothness = get_smoothness(Pos.Screen.xy, Material);
        bool IsMetal, IsHardcodedMetal;
        mat2x3 Reflectance = get_reflectance(Pos.Screen.xy, Material, Color.rgb, IsMetal, IsHardcodedMetal);
        vec2 Lightmap = unpackUnorm2x8(Data2.x);

        // Reflections
        if (Smoothness > 0.3 && isEyeInWater == 0) {
            #ifdef ROUGH_REFLECTIONS
            bool RoughReflections = Smoothness < 0.95 && Material != MATERIAL_WATER;
            #else 
            bool RoughReflections = false;
            #endif

            float RayCount = RoughReflections ? ROUGH_REFLECTIONS_STEPS : 1;
            vec2 Noise = blue_noise(gl_FragCoord.xy, true).rg;
            for(int i = 1; i <= RayCount; i++) {
                vec3 RefNormal;
                if(RoughReflections) {
                    float OffsetAng = (i + Noise.x) * TAU / RayCount;
                    vec2 Offset = Noise.y * vec2(cos(OffsetAng), sin(OffsetAng));
                    float Roughness = smoothness_to_roughness(Smoothness);
                    mat3 TBN = tbn_normal(Normal);
                    RefNormal = TBN * SampleVndf_GGX(Offset * 0.5 + 0.5, -(TBN * Pos.ViewN), vec2(Roughness));
                } else {
                    RefNormal = Normal;
                }
                
                vec3 Dir = reflect(Pos.ViewN, RefNormal);
                vec3 ReflectionBlendFactor = vec3(Smoothness / RayCount);

                if (IsHardcodedMetal) {
                    ReflectionBlendFactor *= fresnel_metals(RefNormal, -Pos.ViewN, Reflectance);
                } else {
                    ReflectionBlendFactor *= schlick(RefNormal, -Pos.ViewN, Reflectance[0]);
                }

                if(IsMetal) {
                    vec3 Albedo = vec3(UnpackX, UnpackY.x);
                    ReflectionBlendFactor *= Albedo;
                }

                Color.rgb += ssr(Dir, Pos.View, Pos.Screen, Lightmap.y, IsHand, Smoothness, Dither) * ReflectionBlendFactor;
            }
        }

        // Specular
        float Shadow = texture(colortex5, Pos.Screen.xy).r;
        if (Shadow != 0) {
            vec3 H = normalize(sLightPosN - Pos.ViewN); // Half-way vector
            vec3 F;
            if (IsHardcodedMetal) {
                F = fresnel_metals(H, -Pos.ViewN, Reflectance);
            }
            else {
                F = schlick(H, -Pos.ViewN, Reflectance[0]);
            }

            vec3 Specular = cook_torrance(-Pos.ViewN, sLightPosN, Normal, smoothness_to_roughness(Smoothness), H, F);

            if(IsMetal) {
                vec3 Albedo = vec3(UnpackX, UnpackY.x);
                Specular *= Albedo;
            }

            float NdotL = max(dot(sLightPosN, Normal), 0);
            Color.rgb += Specular * LightColorDirect * NdotL * Shadow;
        }

        if (!IsHand) {
            float Gtao = 0;
            #ifdef GTAO
                Gtao = gtao(Pos, Normal);
            #endif

            vec3 Rsm = vec3(0);
            #ifdef RSM
                #ifndef DIMENSION_NETHER
                if (!IsMetal)
                    Rsm = rsm(Pos.Player, Normal, LightColorDirect);
                #endif
            #endif
            
            #ifdef SPATIAL_DENOISING
                GISpatial = vec4(Rsm, Gtao);
            #endif

            #ifdef SSAO
                Color.rgb *= 1 - ssao(Color.rgb, Normal, Pos.View);
            #endif
        }
    }
    else {
        Color = texture(colortex0, texcoord);
    }

    #ifdef SPATIAL_DENOISING
        if (Depth >= 1 || IsHand) {
            GISpatial = vec4(0);
        }
    #endif
}
